//
//  PrivacyControlsView.swift
//  death_app Watch App
//
//  Created by Task Master on 2025-09-16.
//

import SwiftUI
import HealthKit

struct PrivacyControlsView: View {
    @State private var showingDeleteAlert = false
    @State private var showingExportSheet = false
    @State private var isDeleting = false
    @State private var isExporting = false
    
    var body: some View {
        List {
            Section("Data Management") {
                Button(action: {
                    showingExportSheet = true
                }) {
                    HStack {
                        Image(systemName: "square.and.arrow.up")
                        Text("Export My Data")
                        Spacer()
                        if isExporting {
                            ProgressView()
                                .scaleEffect(0.8)
                        }
                    }
                }
                .disabled(isExporting)
                
                Button(action: {
                    showingDeleteAlert = true
                }) {
                    HStack {
                        Image(systemName: "trash")
                        Text("Delete All Data")
                        Spacer()
                    }
                    .foregroundColor(.red)
                }
                .disabled(isDeleting)
            }
            
            Section("Privacy Settings") {
                NavigationLink("HealthKit Permissions") {
                    HealthKitPermissionsView()
                }
                
                NavigationLink("Data Retention") {
                    DataRetentionView()
                }
            }
            
            Section("Information") {
                NavigationLink("Privacy Policy") {
                    PrivacyPolicyView()
                }
                
                NavigationLink("Data Usage") {
                    DataUsageView()
                }
            }
        }
        .navigationTitle("Privacy")
        .alert("Delete All Data", isPresented: $showingDeleteAlert) {
            Button("Cancel", role: .cancel) { }
            Button("Delete", role: .destructive) {
                deleteAllUserData()
            }
        } message: {
            Text("This will permanently delete all your personal data, predictions, and settings. This action cannot be undone.")
        }
        .sheet(isPresented: $showingExportSheet) {
            DataExportView()
        }
    }
    
    private func deleteAllUserData() {
        isDeleting = true
        
        Task {
            // Delete UserDefaults
            let defaults = UserDefaults.standard
            let keys = ["hasAcceptedDisclaimer", "userBirthDate", "lastPrediction", "surveyResponses"]
            keys.forEach { defaults.removeObject(forKey: $0) }
            
            // Delete Core Data
            await CoreDataManager.shared.deleteAllData()
            
            // Clear HealthKit authorization (if possible)
            // Note: HealthKit authorization cannot be programmatically revoked
            
            DispatchQueue.main.async {
                isDeleting = false
                // Show success message or dismiss view
            }
        }
    }
}

struct HealthKitPermissionsView: View {
    var body: some View {
        VStack(alignment: .leading, spacing: 15) {
            Text("HealthKit Permissions")
                .font(.headline)
            
            Text("To revoke HealthKit permissions, go to Settings > Privacy & Security > Health > death_app and adjust permissions.")
                .font(.caption)
            
            Text("Note: Permissions cannot be revoked from within the app due to Apple's privacy policies.")
                .font(.caption2)
                .foregroundColor(.secondary)
        }
        .padding()
    }
}

struct DataRetentionView: View {
    @State private var retentionDays = 30
    
    var body: some View {
        VStack {
            Text("Data Retention")
                .font(.headline)
            
            Text("Choose how long to keep your data:")
                .font(.caption)
            
            Picker("Retention Period", selection: $retentionDays) {
                Text("7 days").tag(7)
                Text("30 days").tag(30)
                Text("90 days").tag(90)
                Text("1 year").tag(365)
                Text("Forever").tag(-1)
            }
            .pickerStyle(.wheel)
        }
        .padding()
        .onAppear {
            retentionDays = UserDefaults.standard.integer(forKey: "dataRetentionDays")
            if retentionDays == 0 { retentionDays = 30 }
        }
        .onChange(of: retentionDays) { _, newValue in
            UserDefaults.standard.set(newValue, forKey: "dataRetentionDays")
        }
    }
}

struct PrivacyPolicyView: View {
    var body: some View {
        ScrollView {
            VStack(alignment: .leading, spacing: 15) {
                Text("Privacy Policy")
                    .font(.headline)
                
                Text("Last updated: September 16, 2025")
                    .font(.caption2)
                    .foregroundColor(.secondary)
                
                Text("Data Collection")
                    .font(.subheadline)
                    .fontWeight(.semibold)
                
                Text("We collect minimal data necessary for app functionality including health data you choose to share and basic app usage statistics.")
                    .font(.caption)
                
                Text("Data Usage")
                    .font(.subheadline)
                    .fontWeight(.semibold)
                
                Text("Your data is used solely for providing life expectancy calculations and improving app functionality. We do not share personal data with third parties.")
                    .font(.caption)
                
                Text("Data Storage")
                    .font(.subheadline)
                    .fontWeight(.semibold)
                
                Text("All data is stored locally on your device. No personal information is transmitted to external servers.")
                    .font(.caption)
            }
            .padding()
        }
    }
}

struct DataUsageView: View {
    var body: some View {
        VStack(alignment: .leading, spacing: 10) {
            Text("Data Usage")
                .font(.headline)
            
            Text("This app accesses:")
                .font(.caption)
            
            VStack(alignment: .leading, spacing: 5) {
                Text("• Age and birth date")
                Text("• Health metrics (with permission)")
                Text("• Survey responses")
                Text("• App preferences")
            }
            .font(.caption2)
            .padding(.leading)
            
            Text("All data remains on your device and is never shared externally.")
                .font(.caption2)
                .foregroundColor(.blue)
        }
        .padding()
    }
}

struct DataExportView: View {
    let historicalData: [PredictionHistoryEntry]?
    let trendAnalysis: TrendAnalysis?
    
    @State private var isExporting = false
    @State private var exportComplete = false
    @Environment(\.dismiss) private var dismiss
    
    init(historicalData: [PredictionHistoryEntry]? = nil, trendAnalysis: TrendAnalysis? = nil) {
        self.historicalData = historicalData
        self.trendAnalysis = trendAnalysis
    }
    
    var body: some View {
        VStack(spacing: 20) {
            Text("Export Data")
                .font(.headline)
            
            if isExporting {
                ProgressView("Preparing export...")
            } else if exportComplete {
                Image(systemName: "checkmark.circle.fill")
                    .foregroundColor(.green)
                    .font(.largeTitle)
                Text("Export complete!")
            } else {
                Text("Export all your personal data including predictions, survey responses, and settings.")
                    .font(.caption)
                    .multilineTextAlignment(.center)
                
                Button("Start Export") {
                    startExport()
                }
                .buttonStyle(.borderedProminent)
            }
            
            Button("Close") {
                dismiss()
            }
        }
        .padding()
    }
    
    private func startExport() {
        isExporting = true
        
        Task {
            // Simulate export process
            try await Task.sleep(nanoseconds: 2_000_000_000) // 2 seconds
            
            DispatchQueue.main.async {
                isExporting = false
                exportComplete = true
            }
        }
    }
}

#Preview {
    NavigationView {
        PrivacyControlsView()
    }
}